package com.baidu.nlp.conversation.demo;

import android.content.Context;
import android.media.AudioFormat;
import android.media.AudioManager;
import android.media.AudioTrack;
import android.os.Process;
import android.util.Log;

import java.util.concurrent.BlockingQueue;
import java.util.concurrent.LinkedBlockingQueue;

public class PcmPlayerUtil {

    private static final String TAG = "PcmPlayerUtil";

    // 单例对象
    private static volatile PcmPlayerUtil sInstance;

    public int mSampleRateInHz = 16000;
    public int mChannelConfig = AudioFormat.CHANNEL_OUT_MONO;
    public int mAudioEncoding = AudioFormat.ENCODING_PCM_16BIT;

    private volatile boolean isPlaying = false;
    private volatile boolean mQuit = false;

    private volatile BlockingQueue<byte[]> mTtsRawQueue =
            new LinkedBlockingQueue<>();

    private AudioPlaybackThread mAudioPlayTtsThread;
    public AudioTrack mAudioTrack;

    /**
     * 单例调用
     *
     * @return 单例对象
     */
    public static PcmPlayerUtil getInstance() {
        if (sInstance == null) {
            synchronized (PcmPlayerUtil.class) {
                if (sInstance == null) {
                    sInstance = new PcmPlayerUtil();
                }
            }
        }
        return sInstance;
    }

    public static PcmPlayerUtil getInstance(Context context) {
        if (sInstance == null) {
            synchronized (PcmPlayerUtil.class) {
                if (sInstance == null) {
                    sInstance = new PcmPlayerUtil();
                }
            }
        }
        return sInstance;
    }

    /**
     * 私有构造方法
     */
    private PcmPlayerUtil() {
        init();
    }

    public void init() {
        Log.i(TAG, "init");
        if (mAudioTrack == null) {
            int trackMinSize = AudioTrack.getMinBufferSize(mSampleRateInHz, mChannelConfig, mAudioEncoding);
            mAudioTrack = new AudioTrack(AudioManager.STREAM_MUSIC, mSampleRateInHz, mChannelConfig,
                    mAudioEncoding, trackMinSize, AudioTrack.MODE_STREAM);
            Log.d(TAG, "init,AudioTrack.MinBufferSize=" + trackMinSize);

            // size/（声道数 * 采样位数 / 8  * 采样率）
//            int size = 3000;
//            long time = size/ (2 * 16 / 8 * 16000);

            if (mAudioTrack.getState() == AudioTrack.STATE_INITIALIZED) {
                startAudioPlayTtsThread();
            } else {
                throw new IllegalStateException("AudioTrack initialization was failed!");
            }
        }
    }

    public void start(byte[] rawData, int offset, final int len) {
        if (rawData != null && rawData.length > 0 && len > 0) {
            Log.i(TAG, "start:" + len);
            final byte[] data = new byte[rawData.length - offset];
            System.arraycopy(rawData, offset, data, 0, len);
            mTtsRawQueue.add(data);
            startAudioPlayTtsThread();
        }
    }


    public void stop() {
        Log.i(TAG, "stop");
        try {
            if (mAudioTrack != null) {
                mAudioTrack.pause();
                mAudioTrack.flush();
                mAudioTrack.stop();
            }
        } catch (Exception ex) {
            ex.printStackTrace();
            Log.w(TAG, "stop audiotrack exception:" + ex.getMessage());
        }

        if (mTtsRawQueue != null) {
            mTtsRawQueue.clear();
        }
    }

    public boolean isPlaying() {
        return isPlaying;
    }

    public void clear() {
        if (mTtsRawQueue != null) {
            mTtsRawQueue.clear();
        }
    }

    public void flush() {
        if (mTtsRawQueue != null) {
            mTtsRawQueue.clear();
        }
        try {
            if (mAudioTrack != null) {
                mAudioTrack.flush();
            }
        } catch (Exception ex) {
            ex.printStackTrace();
            Log.w(TAG, "clearTtsData audiotrack exception:" + ex.getMessage());
        }
    }

    public void release() {
        Log.i(TAG, "release");
        mQuit = true;
        stop();
        if (mAudioTrack != null) {
            mAudioTrack.release();
        }
    }


    private void startAudioPlayTtsThread() {
        if (mAudioPlayTtsThread == null || mAudioPlayTtsThread.isFinish) {
            mAudioPlayTtsThread = null;
            mQuit = false;
            mAudioPlayTtsThread = new AudioPlaybackThread();
            mAudioPlayTtsThread.start();
            Log.i(TAG, "startAudioPlayTtsThread");
        }
    }

    class AudioPlaybackThread extends Thread {

        public boolean isFinish = false;

        @Override
        public void run() {
            isFinish = false;
            Process.setThreadPriority(Process.THREAD_PRIORITY_BACKGROUND);
            try {
                if (mAudioTrack != null) {
                    mAudioTrack.play();
                }
            } catch (Exception ex) {
                ex.printStackTrace();
                Log.d(TAG, "AudioPlaybackThread.run() ,AudioPlaybackThread start running:" + ex.getMessage());
            }

            byte[] ttsRawArray = null;
            while (true) {
                if (mQuit) {
                    Log.i(TAG, "AudioPlaybackThread.run() ,AudioPlaybackThread exit play stop");
                    break;
                }

                try {
                    isPlaying = false;
                    ttsRawArray = mTtsRawQueue.take();
                    Log.d(TAG, "AudioPlaybackThread.run() ,mTtsRawQueue.take()");
                } catch (InterruptedException e) {
                    Log.d(TAG, "AudioPlaybackThread.run() ,InterruptedException=" + e.getMessage());
                    if (mQuit) {
                        return;
                    }
                    continue;
                }

                if (ttsRawArray == null || ttsRawArray.length == 0) {
                    continue;
                }
                try {
                    if (mAudioTrack != null) {
                        isPlaying = true;
                        mAudioTrack.write(ttsRawArray, 0, ttsRawArray.length);
                    }
                } catch (Exception ex) {
                    isPlaying = false;
                    ex.printStackTrace();
                    Log.d(TAG, "AudioPlaybackThread.run() ,AudioPlaybackThread Exception:" + ex.getMessage());
                } finally {
                    Log.d(TAG, "AudioPlaybackThread.run()，finally");
                }
            }
            isPlaying = false;
            isFinish = true;
        }
    }

}